using System;
using System.IO;

/** Enigma Decompression and Compression Routines
 * 
 * Ported almost line-by-line from the KENS libraries C source code.
 * */

namespace KensNET
{
	public static class Enigma
	{
		
		/////////////////////////////////////////////////////////////////////////////////////////
		//Program entry point
		public static long EDecomp(string SrcFile, string DstFile, long Pointer, bool padding)
		{
			byte packet_length = 0;
			byte mode = 0;
			short output_repeatcount = 0;
			byte bitmask = 0;
			ushort incrementing_value = 0;
			ushort common_value = 0;
			ushort outvalue = 0;
						
			FileStream infile, outfile;
			int offset = 0;
			byte[] data, outData, paddingData;
			int outoffset = 0;
			bool done = false;
			
			int remaining_bits = 8;
			byte input_buffer;

		// There starts the original code by Nemesis (all Console I/O operations were removed)
			if (!File.Exists(SrcFile))
			{
				//		cout << "\n\nInvalid rom filename!";
				return -1; 
			}
			infile = new FileStream(SrcFile, FileMode.Open);
			data = new byte[infile.Length];
			outData = new byte[0x100000];
			paddingData = new byte[0x1000];
			
			infile.Read(data, 0, data.Length);
			
			offset = (int)Pointer;
			input_buffer = data[offset++];
				
		/////////////////////////////////////////////////////////////////////////////////////////
		//Entry point
			packet_length = getbits(8, data, ref offset, ref input_buffer, ref remaining_bits);
			bitmask = getbits(8, data, ref offset, ref input_buffer, ref remaining_bits);
			incrementing_value = (ushort)(getbits(8, data, ref offset, ref input_buffer, ref remaining_bits) << 8);
			incrementing_value |= getbits(8, data, ref offset, ref input_buffer, ref remaining_bits);
			common_value = (ushort)(getbits(8, data, ref offset, ref input_buffer, ref remaining_bits) << 8);
			common_value |= getbits(8, data, ref offset, ref input_buffer, ref remaining_bits);
		
		/////////////////////////////////////////////////////////////////////////////////////////
		//Main algorithm
			for(;!done;)
			{
				if((getbits(1, data, ref offset, ref input_buffer, ref remaining_bits)) == 1)
				{
					mode = getbits(2, data, ref offset, ref input_buffer, ref remaining_bits);
					switch(mode)
					{
					case 0:
					case 1:
					case 2:
						{
							output_repeatcount = getbits(4, data, ref offset, ref input_buffer, ref remaining_bits);
							outvalue = getvalue(packet_length, data, ref offset, ref input_buffer, ref remaining_bits, bitmask);
							for(;output_repeatcount >= 0; output_repeatcount--)
							{
								outData[outoffset++] = (byte)((outvalue >> 8) & 0x00FF);
								outData[outoffset++] = (byte)(outvalue & 0x00FF);
								
								switch(mode)
								{
								case 0:
									break;
								case 1:
									outvalue++;
									break;
								case 2:
									outvalue--;
									break;
								}
							}
							break;
						}
					case 3:
						{
							output_repeatcount = getbits(4, data, ref offset, ref input_buffer, ref remaining_bits);
							if(output_repeatcount != 0x0F)
							{
								for(;output_repeatcount >= 0; output_repeatcount--)
								{
									outvalue = getvalue(packet_length, data, ref offset, ref input_buffer, ref remaining_bits, bitmask);
									outData[outoffset++] = (byte)((outvalue >> 8) & 0x00FF);
									outData[outoffset++] = (byte)(outvalue & 0x00FF);
								}
							}
							else
							{
								done = true;
							}
							break;
						}
					}
				}
				else
				{
					if((getbits(1, data, ref offset, ref input_buffer, ref remaining_bits)) == 0)
					{
						output_repeatcount = getbits(4, data, ref offset, ref input_buffer, ref remaining_bits);
						for(; output_repeatcount >= 0; output_repeatcount--)
						{
							outData[outoffset++] = (byte)((incrementing_value >> 8) & 0x00FF);
							outData[outoffset++] = (byte)(incrementing_value & 0x00FF);
							incrementing_value++;
						}
					}
					else
					{
						output_repeatcount = getbits(4, data, ref offset, ref input_buffer, ref remaining_bits);
						for(; output_repeatcount >= 0; output_repeatcount--)
						{
							outData[outoffset++] = (byte)((common_value >> 8) & 0x00FF);
							outData[outoffset++] = (byte)(common_value & 0x00FF);
						}
					}
				}
			}
		
		/////////////////////////////////////////////////////////////////////////////////////////
		//Output completed file
			outfile = new FileStream(DstFile, FileMode.Create);
			
			if(padding)
			{
				outfile.Write(paddingData, 0, 0x1000);
				for(int loopcount = 0; loopcount < 0x2000; loopcount += 0x80)
				{
					outfile.Write(paddingData, 0, 0x20);
					outfile.Write(outData, (loopcount/2), 0x40);
					outfile.Write(paddingData, 0, 0x20);
				}
				outfile.Write(paddingData, 0, 0x1000);
			}
			else
			{
				outfile.Write(outData, 0, outoffset);
			}
		
		//	printf("Done!\n\n");
		//	printf("Archive in %s successfully extracted to file %s.\n\n", argv[1], argv[2]);
		//	printf("Compressed file size:\t\t0x%X\t\t%i\n", (offset - originaloffset), (offset - originaloffset));
		//	printf("Uncompressed file size:\t\t0x%X\t\t%i\n", _tell(outfile), _tell(outfile));
		//	printf("Location in file:\t\t0x%X - 0x%X\n", originaloffset, offset);
		//	printf("Compression rate:\t\t%.2f%%\n\n", (100 - (((float)(offset - originaloffset) / (float)(_tell(outfile))) * 100)));
		
			outfile.Close();
			infile.Close();
			return 0;
		}
		
		/////////////////////////////////////////////////////////////////////////////////////////
		//Getvalue function		- This function extracts a 16-bit value from the compressed data.
		private static ushort getvalue(byte packet_length, byte[] data, ref int offset, ref byte input_buffer, ref int remaining_bits, byte bitmask)
		{
			ushort outvalue, addvalue = 0;
			for(int loopcount = 0; loopcount < 5; loopcount++)
			{
				if(((bitmask >> (4 - loopcount)) & 0x01) != 0)
				{
					addvalue |= (ushort)(getbits(1, data, ref offset, ref input_buffer, ref remaining_bits) << (0xF - loopcount));
				}
			}
		
			if(packet_length > 8)
			{
				outvalue = (ushort)(getbits(packet_length - 8, data, ref offset, ref input_buffer, ref remaining_bits) << 8);
				outvalue |= getbits(8, data, ref offset, ref input_buffer, ref remaining_bits);
			}
			else
			{
				outvalue = getbits(packet_length, data, ref offset, ref input_buffer, ref remaining_bits);
			}
		
			outvalue &= (ushort)(0xFFFF ^ (0xFFFF << packet_length));
			outvalue += addvalue;
			return outvalue;
		}
		
		
		private static byte getbits(int number, byte[] input, ref int offset, ref byte input_buffer, ref int remaining_bits)
		{
			byte val = 0;
		
			if(number > remaining_bits)
			{
				val = (byte)(input_buffer >> (8 - number));
				input_buffer = input[offset++];
				val |= (byte)((input_buffer & (0xFF << (8 - number + remaining_bits))) >> (8 - number + remaining_bits));
				input_buffer <<= (number - remaining_bits);
				remaining_bits = 8 - (number - remaining_bits);
			}
			else
			{
				val = (byte)((input_buffer & (0xFF << (8 - number))) >> (8 - number));
				remaining_bits -= number;
				input_buffer <<= number;
			}
		
			return val;
		}
		
		
		/*********************************Variabale block***************************************\
		|	This contains all the global variables that are used throughout the program.
		\*********************************Variabale block***************************************/
		public struct Enigmess
		{
			public bool done;
			public byte bytebuffer;
			public byte[] pointer;
			public short checkbuffer;
			public int loopcount;
			public FileStream infile;
			public FileStream outfile;
			public int infilelength;
			public int offset;
			public int waitingbits;
			
			public int incp;
			public int commonp;
			public int comp;
			public int compinc;
			public int compdec;
			public int comprep;
			
			public byte packet_length;
			public byte bitmask;
			public ushort common_value;
			public ushort tempval;
			public short incrementing_value;
			
			public void Init(){
				done = false;
				bytebuffer = 0;
				checkbuffer = 0;
				offset = 0;
				waitingbits = 0;
				incp = 0;
				commonp = 0;
				comp = 0;
				compinc = 0;
				compdec = 0;
				comprep = 0;
				packet_length = 0x0B;
				bitmask = 0x00;
				common_value = 0x0000;
				tempval = 0x0000;
				incrementing_value = 0x0000;
			}
		}
		
		/***************************************************************************************\
		| Someone forgot to put a comment here! So I added one.
		\***************************************************************************************/
		public static long EComp(string srcFile, string dstFile, bool padding)
		{
			Enigmess dat = new Enigmess();
			dat.Init();
		
		// There starts the original code by Nemesis (all Console I/O opperations were removed)
		/////////////////////////////////////////////////////////////////////////////////////////
		//File handling
			if (!File.Exists(srcFile))
			{
		//		cout << "\n\nError opening source file!";
				return -1;
			}
			dat.infile = new FileStream(srcFile, FileMode.Open);
			dat.outfile = new FileStream(dstFile, FileMode.Create);
			dat.infilelength = (int)dat.infile.Length;
			dat.pointer = new byte[dat.infilelength];
		
		/////////////////////////////////////////////////////////////////////////////////////////
		//Format conversion		- This section of code removes the surrounding areas if the stage
		//						- is stored in the full 80x80 block version
			if(padding)
			{
				dat.infile.Seek(0x1000, SeekOrigin.Begin);
				for(dat.loopcount = 0; dat.loopcount < 0x2000; dat.loopcount += 0x80)
				{
					dat.infile.Seek(dat.infile.Position+0x20, SeekOrigin.Begin);
					byte[] tmpData = new byte[0x40];
					dat.infile.Read(tmpData, 0, 0x40);
					Array.Copy(tmpData, 0, dat.pointer, (dat.loopcount / 2), 0x40);
					dat.infile.Seek(dat.infile.Position+0x20, SeekOrigin.Begin);
				}
				dat.infilelength = 0x1000;
			}
			else
			{
				dat.infile.Read(dat.pointer, 0, dat.infilelength);
			}
		
			dat.tempval = dat.pointer[0];
			ushort tempval2 = 0;
			for(dat.loopcount = 0; dat.loopcount < dat.infilelength; dat.loopcount += 2)
			{
				tempval2 |= (ushort)(dat.pointer[dat.loopcount] << 8);
				tempval2 |= (ushort)dat.pointer[dat.loopcount + 1];
			}
		
			dat.bitmask = (byte)(tempval2 >> 0x0B);
			for(dat.loopcount = 0; dat.loopcount < 0x0A; dat.loopcount++)
			{
				if(((tempval2 >> (0x0A - dat.loopcount)) & 0x0001) == 0)
				{
					dat.packet_length--;
				}
				else
				{
					dat.loopcount = 0x0A;
				}
			}
		
		//	printf("Calculating common and increment values......");
			calc_headerval(ref dat);
		//	printf("\tDone!\n");
			byte crapway1 = (byte)(dat.incrementing_value >> 8);
			byte crapway2 = (byte)(dat.incrementing_value & 0x00FF);
			byte crapway3 = (byte)(dat.common_value >> 8);
			byte crapway4 = (byte)(dat.common_value & 0x00FF);
			dat.outfile.WriteByte(dat.packet_length);
			dat.outfile.WriteByte(dat.bitmask);
			dat.outfile.WriteByte(crapway1);
			dat.outfile.WriteByte(crapway2);
			dat.outfile.WriteByte(crapway3);
			dat.outfile.WriteByte(crapway4);
			dat.bitmask <<= 3;
		
		/////////////////////////////////////////////////////////////////////////////////////////
		//Main compression loop
		//	printf("Compressing file......");
			for(;!dat.done;)
			{
				if(dat.offset >= dat.infilelength)
				{
					writebit(1, ref dat);
					writebit(1, ref dat);
					writebit(1, ref dat);
					writebit(1, ref dat);
					writebit(1, ref dat);
					writebit(1, ref dat);
					writebit(1, ref dat);
					dat.bytebuffer <<= (8 - dat.waitingbits);
					dat.outfile.WriteByte(dat.bytebuffer);
					dat.done = true;
					continue;
				}
				
				for(dat.commonp = 0; (dat.offset + (dat.commonp * 2) < dat.pointer.Length) && (checkval(dat.commonp * 2, ref dat) == dat.common_value) && (dat.commonp < 0x10); dat.commonp++){}
				for(dat.incp = 0; (dat.offset + (dat.incp * 2) < dat.pointer.Length) && (checkval(dat.incp * 2, ref dat) == (dat.incrementing_value + dat.incp)) && (dat.incp < 0x10); dat.incp++){}
				for(dat.compinc = 0; (dat.offset + (dat.compinc * 2) < dat.pointer.Length) && (checkval(dat.compinc * 2, ref dat) == (checkval(0, ref dat) + dat.compinc)) && (dat.compinc < 0x10); dat.compinc++){}
				for(dat.compdec = 0; (dat.offset + (dat.compdec * 2) < dat.pointer.Length) && (checkval(dat.compdec * 2, ref dat) == (checkval(0, ref dat) - dat.compdec)) && (dat.compdec < 0x10); dat.compdec++){}
				for(dat.comp = 0; (dat.offset + (dat.comp * 2) < dat.pointer.Length) && (checkval(dat.comp * 2, ref dat) == checkval(0, ref dat)) && (dat.comp < 0x10); dat.comp++){}
				for(dat.comprep = 0; (dat.offset + (dat.comprep * 2) < dat.pointer.Length) && (dat.comprep < 0x0F) && (checkval(dat.comprep * 2, ref dat) != dat.common_value) && (checkval(dat.comprep * 2, ref dat) != dat.incrementing_value) && (checkval(dat.comprep * 2, ref dat) != checkval(dat.comprep * 2 + 2, ref dat)) && (checkval(dat.comprep * 2, ref dat) != (checkval(dat.comprep * 2 + 2, ref dat) + 1)) && (checkval(dat.comprep * 2, ref dat) != (checkval(dat.comprep * 2 + 2, ref dat) - 1)); dat.comprep++){}
		
				if(dat.incp > 0)
				{
					writeinc(dat.incp, ref dat);
					dat.incrementing_value += (short)dat.incp;
					continue;
				}
				if(dat.commonp > 0)
				{
					writecommon(dat.commonp, ref dat);
					continue;
				}
				if((dat.comp > dat.compinc) && (dat.comp > dat.compdec))
				{
					writecomp(dat.comp, ref dat);
					continue;
				}
				if(dat.compinc > dat.compdec)
				{
					writecompinc(dat.compinc, ref dat);
					continue;
				}
				if(dat.compdec > 1)
				{
					writecompdec(dat.compdec, ref dat);
					continue;
				}
				writecomprep(dat.comprep, ref dat);
			}
		
		//	printf("\t\t\t\tDone!\n\n%s successfully compressed into file %s\nOriginal filesize:\t%i\nCompressed filesize:\t%i\nPercentage of original:\t%i", argv[1], argv[2], dat.infilelength, _tell(outfile), (int)(((float)_tell(outfile) / (float)dat.infilelength) * 100));
		
			if ((dat.outfile.Position & 1) > 0) dat.outfile.Seek(1, SeekOrigin.Current);
		
			dat.infile.Close();
			dat.outfile.Close();
			return 0;
		}
		
		/*********************************Read/Write functions**********************************\
		|	These functions give a layer of abstraction over the reading and writing to and from
		|	the files.
		\*********************************Read/Write functions**********************************/
		public static byte writebit(byte bit, ref Enigmess dat)
		{
			dat.waitingbits++;
			dat.bytebuffer <<= 1;
			dat.bytebuffer |= bit;
		
			if(dat.waitingbits >= 8)
			{
				dat.outfile.WriteByte(dat.bytebuffer);
				dat.waitingbits = 0;
			}
			return bit;
		}
		
		public static short checkval(int offsetloc, ref Enigmess dat)
		{
			dat.checkbuffer = (short)(dat.pointer[dat.offset + offsetloc] << 8);
			dat.checkbuffer |= dat.pointer[dat.offset + offsetloc + 1];
			return dat.checkbuffer;
		}
		
		/**********************************Packet write functions*******************************\
		|	These functions write the compressed packets to the output file.
		\**********************************Packet write functions*******************************/
		public static int writeinc(int vincp, ref Enigmess dat)
		{
			vincp--;
			writebit(0, ref dat);
			writebit(0, ref dat);
			writebit((byte)((vincp & 0x0008) >> 3), ref dat);
			writebit((byte)((vincp & 0x0004) >> 2), ref dat);
			writebit((byte)((vincp & 0x0002) >> 1), ref dat);
			writebit((byte)(vincp & 0x0001), ref dat);
			dat.offset += ((vincp * 2) + 2);
		
			return 0;
		}
		
		public static int writecommon(int vcommonp, ref Enigmess dat)
		{
			vcommonp--;
			writebit(0, ref dat);
			writebit(1, ref dat);
			writebit((byte)((vcommonp & 0x0008) >> 3), ref dat);
			writebit((byte)((vcommonp & 0x0004) >> 2), ref dat);
			writebit((byte)((vcommonp & 0x0002) >> 1), ref dat);
			writebit((byte)(vcommonp & 0x0001), ref dat);
			dat.offset += ((vcommonp * 2) + 2);
		
			return 0;
		}
		
		public static int writecomp(int vcomp, ref Enigmess dat)
		{
			vcomp--;
			writebit(1, ref dat);
			writebit(0, ref dat);
			writebit(0, ref dat);
			writebit((byte)((vcomp & 0x0008) >> 3), ref dat);
			writebit((byte)((vcomp & 0x0004) >> 2), ref dat);
			writebit((byte)((vcomp & 0x0002) >> 1), ref dat);
			writebit((byte)(vcomp & 0x0001), ref dat);
		
			for(dat.loopcount = 0; dat.loopcount < 5; dat.loopcount++)
			{
				if(((dat.bitmask << dat.loopcount) & 0x80) != 0)
				{
					writebit((byte)((checkval(0, ref dat) >> (0x0F - dat.loopcount)) & 0x0001), ref dat);
				}
			}
			for(dat.loopcount = 0; dat.loopcount < dat.packet_length; dat.loopcount++)
			{
				writebit((byte)((checkval(0, ref dat) >> ((dat.packet_length - 1) - dat.loopcount)) & 0x0001), ref dat);
			}
		
			dat.offset += ((vcomp * 2) + 2);
			return 0;
		}
		
		public static int writecompinc(int vcompinc, ref Enigmess dat)
		{
			vcompinc--;
			writebit(1, ref dat);
			writebit(0, ref dat);
			writebit(1, ref dat);
			writebit((byte)((vcompinc & 0x0008) >> 3), ref dat);
			writebit((byte)((vcompinc & 0x0004) >> 2), ref dat);
			writebit((byte)((vcompinc & 0x0002) >> 1), ref dat);
			writebit((byte)(vcompinc & 0x0001), ref dat);
			
			for(dat.loopcount = 0; dat.loopcount < 5; dat.loopcount++)
			{
				if(((dat.bitmask << dat.loopcount) & 0x80) != 0)
				{
					writebit((byte)((checkval(0, ref dat) >> (0x0F - dat.loopcount)) & 0x0001), ref dat);
				}
			}
			for(dat.loopcount = 0; dat.loopcount < dat.packet_length; dat.loopcount++)
			{
				writebit((byte)((checkval(0, ref dat) >> ((dat.packet_length - 1) - dat.loopcount)) & 0x0001), ref dat);
			}
		
			dat.offset += ((vcompinc * 2) + 2);
			return 0;
		}
		
		public static int writecompdec(int vcompdec, ref Enigmess dat)
		{
			vcompdec--;
			writebit(1, ref dat);
			writebit(1, ref dat);
			writebit(0, ref dat);
			writebit((byte)((vcompdec & 0x0008) >> 3), ref dat);
			writebit((byte)((vcompdec & 0x0004) >> 2), ref dat);
			writebit((byte)((vcompdec & 0x0002) >> 1), ref dat);
			writebit((byte)(vcompdec & 0x0001), ref dat);
			
			for(dat.loopcount = 0; dat.loopcount < 5; dat.loopcount++)
			{
				if(((dat.bitmask << dat.loopcount) & 0x80) != 0)
				{
					writebit((byte)((checkval(0, ref dat) >> (0x0F - dat.loopcount)) & 0x0001), ref dat);
				}
			}
			for(dat.loopcount = 0; dat.loopcount < dat.packet_length; dat.loopcount++)
			{
				writebit((byte)((checkval(0, ref dat) >> ((dat.packet_length - 1) - dat.loopcount)) & 0x0001), ref dat);
			}
		
			dat.offset += ((vcompdec * 2) + 2);
			return 0;
		}
		
		public static int writecomprep(int vcomprep, ref Enigmess dat)
		{
			vcomprep--;
			writebit(1, ref dat);
			writebit(1, ref dat);
			writebit(1, ref dat);
			writebit((byte)((vcomprep & 0x0008) >> 3), ref dat);
			writebit((byte)((vcomprep & 0x0004) >> 2), ref dat);
			writebit((byte)((vcomprep & 0x0002) >> 1), ref dat);
			writebit((byte)(vcomprep & 0x0001), ref dat);
		
			for(; vcomprep >= 0; vcomprep--)
			{
				for(dat.loopcount = 0; dat.loopcount < 5; dat.loopcount++)
				{
					if(((dat.bitmask << dat.loopcount) & 0x80) != 0)
					{
						writebit((byte)((checkval(0, ref dat) >> (0x0F - dat.loopcount)) & 0x0001), ref dat);
					}
				}
				for(dat.loopcount = 0; dat.loopcount < dat.packet_length; dat.loopcount++)
				{
					writebit((byte)((checkval(0, ref dat) >> ((dat.packet_length - 1) - dat.loopcount)) & 0x0001), ref dat);
				}
				dat.offset += 2;
			}
			return 0;
		}
		
		/****************************calc_headerval function************************************\
		|	This function calculates the most common value occuring in the file, and the best
		|	increment value to use.
		\****************************calc_headerval function************************************/
		public static void calc_headerval(ref Enigmess dat)
		{
			int comhitcount = 0;
			int inchitcount = 0;
			int comhighestcount = 0;
			int inchighestcount = 0;
			int commonoffset = 0;
			ushort commontemp = 0;
			ushort tempincval = 0;
			ushort lowestval = 0xFFFF;
			for(dat.loopcount = 0; dat.loopcount < dat.infilelength; dat.loopcount += 2)
			{
				commontemp = (ushort)(dat.pointer[commonoffset++] << 8);
				commontemp |= dat.pointer[commonoffset++];
				if(commontemp < lowestval)
				{
					lowestval = commontemp;
				}
			}
		
			for(dat.loopcount = lowestval; dat.loopcount < 0x10000; dat.loopcount++)
			{
				commonoffset = 0;
				inchitcount = 0;
				comhitcount = 0;
				tempincval = (ushort)dat.loopcount;
				for(;commonoffset < dat.infilelength;)
				{
					commontemp = (ushort)(dat.pointer[commonoffset++] << 8);
					commontemp |= dat.pointer[commonoffset++];
					if(commontemp == (ushort)dat.loopcount)
					{
						comhitcount++;
					}
					if(commontemp == tempincval)
					{
						inchitcount++;
						tempincval++;
					}
					if(comhitcount > comhighestcount)
					{
						comhighestcount++;
						dat.common_value = (ushort)dat.loopcount;
					}
					if(inchitcount > inchighestcount)
					{
						inchighestcount++;
						dat.incrementing_value = (short)(tempincval - inchitcount);
					}
				}
			}
		}
		
	}
}

