using System;
using System.IO;

/** Kosinski Decompression and Compression Routines
 * 
 * Ported almost line-by-line from the KENS libraries C source code.
 * */

namespace KensNET
{
	public class Kosinski
	{
		public Kosinski (){}
		
		public enum KResult{
			SUCCESS, ERROR_UNKNOWN, ERROR_FILE_NOT_FOUND, ERROR_MODULED_GREATHER_THAN_65535
		}
		
		public static KResult KDecomp(string srcFile, string dstFile, long location, bool Moduled)
		{
			// Files
			FileStream src, dst;
			
			// Bitfield Infos
			ushort BITFIELD;
			byte BFP;
			byte Bit;
			
			// R/W infos
			byte Byte;
			byte Low, High;
			
			// Localization Infos
			int Pointer;
				
			// Count and Offest
			long Count = 0;
			long Offset = 0;
			
			// Moduled infos
			long FullSize = 0;
			long DecBytes = 0;
			
			//------------------------------------------------------------------------------------------------
			if (!File.Exists(srcFile)) return KResult.ERROR_FILE_NOT_FOUND;
			src = new FileStream(srcFile, FileMode.Open);
			dst = new FileStream(dstFile, FileMode.Create);
			
			src.Seek(location, SeekOrigin.Begin);
			
			if (Moduled)
			{
				High = (byte)src.ReadByte();
				Low = (byte)src.ReadByte();
				FullSize = ((long)High << 8) + (long)Low;
			}
		
			start:
				BITFIELD = ReadShort(src);
				BFP=0;
			
			//------------------------------------------------------------------------------------------------
				while(true)
				{
					if((BITFIELD & (1<<BFP++)) > 0) Bit=1; else Bit=0;
					if (BFP>=16) { BITFIELD=ReadShort(src); BFP=0; }
			//-- Direct Copy ---------------------------------------------------------------------------------
					if (Bit > 0)
					{
						dst.WriteByte((byte)src.ReadByte());
						DecBytes+=1;
					}
					else
					{
						if((BITFIELD & (1<<BFP++))>0) Bit=1; else Bit=0;
						if (BFP>=16) { BITFIELD = ReadShort(src); BFP=0; }
			//-- Embedded / Separate -------------------------------------------------------------------------
						if (Bit > 0)
						{
							Low = (byte)src.ReadByte();
							High = (byte)src.ReadByte();
			
							Count=(long)(High & 0x07);
			
							if (Count==0)
							{
								Count = (byte)src.ReadByte();
								if (Count==0) break;
								if (Count==1) continue;
							}
							else
							{
								Count+=1;
						    }
			
							Offset = 0xFFFFE000 | ((long)(0xF8 & High) << 5) | (long)Low;
						}
			//-- Inline --------------------------------------------------------------------------------------
						else
						{
							if((BITFIELD & (1<<BFP++))>0) Low=1; else Low=0;
							if (BFP>=16) { BITFIELD=ReadShort(src); BFP=0; }
							if((BITFIELD & (1<<BFP++))>0) High=1; else High=0;
							if (BFP>=16) { BITFIELD = ReadShort(src); BFP=0; }
			
							Count = ((long)Low)*2 + ((long)High) + 1;
			
							Offset = src.ReadByte();
							Offset|=0xFFFFFF00;
						}
			//-- Write to file for indirect copy -------------------------------------------------------------
						if(Offset > 0x80000000) Offset-=0x100000000; //Longs are 64-bit in C#. Negate them manually
						for (int i=0; i<=Count; i++)
						{
							Pointer=(int)dst.Position;
							dst.Seek(Offset, SeekOrigin.Current);
							Byte = (byte)dst.ReadByte();
							dst.Seek(Pointer, SeekOrigin.Begin);
							dst.WriteByte(Byte);
						}
						DecBytes+=Count+1;
			//------------------------------------------------------------------------------------------------
					}
				}
			//------------------------------------------------------------------------------------------------
			
//			end:
				if (Moduled)
				{
					if (DecBytes < FullSize)
					{
						do { Byte=(byte)src.ReadByte(); } while (Byte==0);
						src.Seek(-1, SeekOrigin.Current);
						goto start;
					}
				}
				dst.Close();
				src.Close();
				return KResult.SUCCESS;
			}
		
		/// <summary>
		/// Read a short as little endian
		/// </summary>
		private static ushort ReadShort(FileStream strm){
			byte[] arr = new byte[2];
			strm.Read(arr, 0, 2);
			return (ushort)((arr[0]) + (arr[1] << 8)); 
		}
		
		/// <summary>
		/// Kosinski compression, using the default Slide Window and RecLen values
		/// </summary>
		public static KResult KComp(string srcFile, string dstFile, bool moduled){
			return KComp(srcFile, dstFile, 8192, 256, moduled);
		}
		
		//-----------------------------------------------------------------------------------------------
		// Name: KComp(char *SrcFile, char *DstFile, int SlideWin, int RecLen, bool Moduled)
		// Desc: Compresses the data using the Kosinski compression format
		//-----------------------------------------------------------------------------------------------
		public static KResult KComp(string srcFile, string dstFile, int slideWin, int recLen, bool moduled)
		{
		// Files
			FileStream src;
			FileStream dst;
		
		// Pre-write infos
			ushort BITFIELD;
			byte BFP;
			byte[] Data = new byte[64];
			byte DS;
		
		// Buffer Infos
			byte[] Buffer;
			int BSize;
			int BPointer;
			int bufferOffset=0;
			
		// Count and Offest
			int ICount = 0;
			int IOffset = 0;
		
		// Temp Info and Offest for writing
			ushort Info;
			ushort Off;
		
		// Counters
			int i, imax, j=0, k;
		
		// Moduled infos
			int FullSize=0;
			int CompBytes = 0;
			byte High, Low;
		
		//----------------------------------------------------------------------------------------------------------------
		
			if(!File.Exists(srcFile)) return KResult.ERROR_FILE_NOT_FOUND;
			src = new FileStream(srcFile, FileMode.Open);
			BSize=(int)src.Length;
			Buffer = new byte[BSize];
			src.Read(Buffer, 0, BSize);
			src.Close();
			dst = new FileStream(dstFile, FileMode.Create);
			
			BITFIELD=1;
			BFP=1;
			Data[0]=Buffer[0];
			BPointer=1;
			DS=1;
		
			if (moduled)
			{
				if (BSize>65535) { dst.Close(); return KResult.ERROR_MODULED_GREATHER_THAN_65535; }
				FullSize=BSize;
				if (BSize>0x1000) BSize=0x1000;
				High = (byte)(FullSize >> 8);
				Low = (byte)(FullSize & 0xFF);
				dst.WriteByte(High);
				dst.WriteByte(Low);
			}
		
		//----------------------------------------------------------------------------------------------------------------
		
		start:
			ICount=recLen; if (BSize-BPointer<recLen) { ICount=BSize; ICount-=BPointer; }
			k=1;
			imax = BPointer; imax-=slideWin; if (imax<0) imax=0;
			i=BPointer; --i;
		
			do {
				j=0;
				while ( Buffer[bufferOffset+i+j] == Buffer[bufferOffset+BPointer+j] ) { if(++j>=ICount) break; }
				if (j>k) { k=j; IOffset=i; }
			} while (i-->imax);
			ICount=k;
		
		//----------------------------------------------------------------------------------------------------------------
		
		//write:
			if (ICount==1)
			{
				BITFIELD|=(ushort)(1<<BFP); // 2^BFP
				if (++BFP==16) { WriteShort(dst, BITFIELD); dst.Write(Data, 0, DS); BITFIELD=BFP=DS=0; }	
				
				Data[DS]=Buffer[bufferOffset+BPointer];
				DS+=1;
			}
			else if ( (ICount==2) & (BPointer-IOffset>256) )
			{
				BITFIELD|=(ushort)(1<<BFP); // 2^BFP
				if (++BFP==16) { WriteShort(dst, BITFIELD); dst.Write(Data, 0, DS); BITFIELD=BFP=DS=0; }
		
				Data[DS]=Buffer[bufferOffset+BPointer];
				DS+=1;
		
				--ICount;
			}
			else if ( (ICount < 6) & (BPointer-IOffset<=256) )
			{
				if (++BFP==16) { WriteShort(dst, BITFIELD); dst.Write(Data, 0, DS); BITFIELD=BFP=DS=0; }
		
				if (++BFP==16) { WriteShort(dst, BITFIELD); dst.Write(Data, 0, DS); BITFIELD=BFP=DS=0; }
		
				if ( ((ICount-2) & 0x02) == 0x02) BITFIELD|=(ushort)(1<<BFP); // 2^BFP
				if (++BFP==16) { WriteShort(dst, BITFIELD); dst.Write(Data, 0, DS); BITFIELD=BFP=DS=0; }
		
				if ( ((ICount-2) & 0x01) == 0x01) BITFIELD|=(ushort)(1<<BFP); // 2^BFP
				if (++BFP==16) { WriteShort(dst, BITFIELD); dst.Write(Data, 0, DS); BITFIELD=BFP=DS=0; }
		
				Data[DS]=(byte)(~(BPointer-IOffset-1));
				DS+=1;
			}
		
			else
			{
				if (++BFP==16) { WriteShort(dst, BITFIELD); dst.Write(Data, 0, DS); BITFIELD=BFP=DS=0; }
		
				BITFIELD|=(ushort)(1<<BFP); // 2^BFP
				if (++BFP==16) { WriteShort(dst, BITFIELD); dst.Write(Data, 0, DS); BITFIELD=BFP=DS=0; }
		
				if (ICount-2<8)
				{
					Off = (ushort)(BPointer-IOffset-1);
					Info = (ushort)( ( ~( (Off << 8) | (Off >> 5) ) & 0xFFF8 ) | ((byte)(ICount - 2)) );
					Data[DS] = (byte)( Info >> 8 );
					Data[DS+1] = (byte)( Info & 0x00FF );
					DS+=2;
				}
				else
				{
					Off = (ushort)(BPointer-IOffset-1);
					Info = (ushort)( ~( (Off << 8) | (Off >> 5) ) & 0xFFF8 );
					Data[DS] = (byte)( Info >> 8 );
					Data[DS+1] = (byte)( Info & 0x00FF );
					Data[DS+2] = (byte)(ICount - 1);
					DS+=3;
				}
			}
		
		//----------------------------------------------------------------------------------------------------------------
		
			BPointer+=ICount;
			if (BPointer<BSize) goto start;
		
		//----------------------------------------------------------------------------------------------------------------
			if (++BFP==16) { WriteShort(dst, BITFIELD); dst.Write(Data, 0, DS); BITFIELD=BFP=DS=0; }
		
			BITFIELD|=(ushort)(1<<BFP); // 2^BFP
			if (++BFP==16) { WriteShort(dst, BITFIELD); dst.Write(Data, 0, DS); BITFIELD=BFP=DS=0; }
		
			Data[DS] = 0x00;
			Data[DS+1] = 0xF0;
			Data[DS+2] = 0x00;
			DS+=3;
		
			WriteShort(dst, BITFIELD); dst.Write(Data, 0, DS); BITFIELD=BFP=DS=0;
		
			if (moduled)
			{
				dst.Seek(16-(dst.Position-2)%16, SeekOrigin.Current);
				CompBytes+=BSize;
				bufferOffset+=BSize;
				if (CompBytes<FullSize)
				{
					BSize=0x1000;
					if (FullSize-CompBytes<0x1000) BSize=FullSize-CompBytes;
					BITFIELD=1;
					BFP=1;
					Data[0]=Buffer[bufferOffset+0];
					BPointer=1;
					DS=1;
					goto start;
				}
			}
		
			if ((dst.Position & 1) > 0) dst.Seek(1, SeekOrigin.Current);
			dst.Close();
			
			return KResult.SUCCESS;
		}
		
		/// <summary>
		/// Read a short as little endian
		/// </summary>
		private static void WriteShort(FileStream strm, ushort val){
			strm.WriteByte((byte)(val&0xFF));
			strm.WriteByte((byte)((val>>8)&0xFF)); 
		}
	}
}

